/*jshint esversion: 6 */

define([
	"lodash", "src/utils", "immutable",
	"src/math/Mat3", "src/math/Vec2", 
	"lib/tasks/dofs",
	"lib/tasks/internal",
	"lib/dev/internal",
	"lib/dev/config",
	],
function(
	lodash, utils, immutable,
	mat3, vec2, 
	dofs,
	internal,
	devInternal,
	config
) {

"use strict";

var getJointHandle = internal.getJointHandle;

if (config.MutableTree.enabled) {
return (function () {
/*============================================================================
	Helpers
============================================================================*/

// FIXME: if it shows up in the profiler move into c++
function matrixFixedPoint (prev, now) {
	// Tuned for Gene by setting values that don't lead to return to rest pop.
	// That test shoulde be performed in the absence of all other behaviors.
	var	translationEps = 0.1,
		linearEps = 0.05;

	var delta;

	// translation
	delta = prev[6] - now[6];
	if (delta < -translationEps || delta > translationEps) return false;

	delta = prev[7] - now[7];
	if (delta < -translationEps || delta > translationEps) return false;

	// linear
	delta = prev[0] - now[0];
	if (delta < -linearEps || delta > linearEps) return false;

	delta = prev[1] - now[1];
	if (delta < -linearEps || delta > linearEps) return false;

	delta = prev[3] - now[3];
	if (delta < -linearEps || delta > linearEps) return false;

	delta = prev[4] - now[4];
	if (delta < -linearEps || delta > linearEps) return false;

	return true;
}

function layerFixedPoint (layer, tomPrev, tomNow) {
	var tree = layer.getHandleTreeArray(),
		aFrameNow = internal.gatherHandleFrames(tree, tomNow.value),
		aFramePrev = internal.gatherHandleFrames(tree, tomPrev.value),
		aLeafRef = tree.getLeafRefArray();

	var	ref = aLeafRef.length;
	while (--ref >= 0)	{
		var refLeaf = aLeafRef[ref];

		if (false === matrixFixedPoint(aFramePrev[refLeaf].matrix, aFrameNow[refLeaf].matrix)) {
			// console.logToUser(`${aFramePrev[refLeaf].matrix} !== ${aFrameNow[refLeaf].matrix}`);
			return false;
		}
	}

	return true;
}

function forEachWarperLayer (fn) {
	// function (layer, tom_1, tom_2, ...) -> function (layer, [tom_1, tom_2, ...])
	var visit = lodash.spread(fn, 1);

	function walk (layer, toms) {
		var puppet = layer.getPuppet(),
			layers = puppet.getLayers(),
			tomKids = toms.map(x => x.children),
			index = -1,
			size = layers.length;

		while (++index < size) {
			var subLayer = layers[index];
			if (subLayer.getPuppet()) {
			if (subLayer.getWarpWithParent()) {
					walk(subLayer, toms);
				} else {
					visit(subLayer, tomKids.map(forest => forest[subLayer.getStageId()]));
				}
			}
		}
	}

	// function (layer, [tom_1, tom_2, ...]) -> function (layer, tom_1, tom_2, ...)
	return lodash.rest(walk);
}

function dofMatrixFromHandle (handle) {
	return new dofs.Matrix({ 
		type : handle.getAutoAttribute(),
		matrix : mat3().setIdentity()
	});
}


// Set enabled to true to report when Layer reaches fixed point and stops warping.
var reportLayerWarps = { enabled : false },
	tLayerState = {};

devInternal.ifEnabled(reportLayerWarps, {

	count (layer) {
		tLayerState[layer.getStageId()] += 1;
	},

	show (layer, aPuppetMatLeaf) {
		var warpCount = tLayerState[layer.getStageId()];
		tLayerState[layer.getStageId()] = 0;

		if (warpCount > 0) {
			console.log("layer '" + layer.getName() + "' stops:\n" + aPuppetMatLeaf);
		}
	}
});


/*
 * The origin handle defines the root frame for warping by its descendent leaf handles.
 * Returned object contains (1) value: the motion of that root,
 * and (2) remove/add: the change of coordinates needed to remove and add that motion from/to descendent handles.
 * 
 * TODO: Long-term gatherHandleFrames (and related functions) should support this functionality.
 * For now, we patch it by explicitlity removing/adding to retain the invariant of compute these frames relative to the Puppet frame.
 */
function warperRootFrame (frameRoot) {
	var value = frameRoot.matrix,
		remove, add;

	try {
		remove = mat3.invert(value);
		add = value;

		return {
			value, add, remove
		};
	} catch (e) {
		// we cannot change coordinates to remove singular root frame
		return {
			value,
			add : false, // signals singular root frame
			remove : mat3.identity() // allows non-distructive removal i.e. when multplying with .remove value
		};
	}
}

						 

var LayerMatrix = {

	init : function init (layer) {
		function initTree (layer, parent0) {
			var handleArray = layer.getHandleTreeArray().aHandle;

			var aDof = handleArray.map(dofMatrixFromHandle);
			
			var newTree = new dofs.Tree({ parent0 }, aDof);

			var tomKids = newTree.children;

			forEachWarperLayer(function (subLayer) {
				tomKids[subLayer.getStageId()] = initTree(subLayer, newTree);
			})(layer);

			return newTree;
		}
								 
		return initTree(layer);
	},

	// Layer ToM? ToM ToM -> ToM
	warp : function warp (layer, tomPrevPrev, tomPrev, tomNow) {

		var lomNow = tomNow.value;

		// initialize assuming warp won't change the new set of dofs
		var aFrameNext = false,
			warperRoot = false,
			warperRootValueNow = false,
			didWarpRun = false;

		// skip warp if there is no new work and warper has reached the fixed point
		if (tomNow.hasChangedSinceLastIncrement() || 
			layerFixedPoint(layer, tomPrevPrev, tomPrev) === false) {

			var puppet = layer.getPuppet(),
				cWarper = layer.getWarperContainer();

			utils.assert(puppet && cWarper, "no warping container.");

			var	tree = layer.getHandleTreeArray(),
				aLeafRef = tree.getLeafRefArray();

			aFrameNext = internal.gatherHandleFrames(tree, lomNow);      // FIXME: "next" to "now"

			// Same goals as code in the comments but faster performance:
			// var	aMatLeaf = lodash(aPuppetFrames).at(aLeafRef).map(lodash.property("matrix")).value(),
			// 		aTypeLeaf = lodash(aPuppetFrames).at(aLeafRef).map(lodash.property("type")).value();
			var	nLeaf = aLeafRef.length,
				aMatLeafNext = new Array(nLeaf),
				aMatAnchorNext = new Array(nLeaf),
				aTypeLeafNext = new Array(nLeaf);

			aLeafRef.forEach(function (ref, i) {
				var frameNext = aFrameNext[ref];

				aMatLeafNext[i] = frameNext.matrix;
				aMatAnchorNext[i] = frameNext.anchor;
				aTypeLeafNext[i] = frameNext.type;
			});

			warperRoot = warperRootFrame(aFrameNext[0]);

			// Record current matrix for later checking whether root (i.e. non-leaf) dof changes.
			warperRootValueNow = cWarper.getMatrix();
			cWarper.setMatrix(warperRoot.value);

			// skip warp for singular root frame
			if (warperRoot.add !== false) {
				// temporarily remove warper root value from leaf handles...
				var	aMat_Handle = aMatLeafNext.map(function (matContainer_Handle) {
						return mat3.multiply(warperRoot.remove, matContainer_Handle);
					});

				cWarper.warp(aMat_Handle, aTypeLeafNext);
				didWarpRun = true;

				// ...and add back warper root value so that state mirrors accumulated values of all handles
				aMat_Handle.forEach(function (mi, i) {
					mat3.multiply(warperRoot.add, mi, aMatLeafNext[i]);
				});

				// fast special purpose scatter that reuses already computed values for leaf parents
				internal.scatterLeafFramesMutable(tree, aFrameNext, aMatLeafNext, lomNow);
			}
		}

		// Update attachment cooordinates when handle dofs change.
		// The first condition takes care of initialization case: container matrix does not yet mirror the value of the root handle.
		if ( (warperRootValueNow && !mat3.equalsApproximately(warperRootValueNow, warperRoot.value)) || didWarpRun ) {

			forEachWarperLayer(function (subLayer) {

				var matWarper_Joint;

				const joint = getJointHandle(subLayer);

				if (joint) {
					// Calculate sub-puppet position from attach-to handle.
					const tree = layer.getHandleTreeArray(),
						jointRef = tree.getHandleRef(joint);

					matWarper_Joint = aFrameNext[jointRef].matrix;
				} else {
					// Calculate sub-puppet position from attach-to coordinates.
					matWarper_Joint = layer.getWarpAtAttachment(aMatLeafNext, subLayer);
				}

				subLayer.filterJointFrame(matWarper_Joint, matWarper_Joint);

				var parentPuppet = subLayer.parentPuppet,
					matParentPuppet_Warper = parentPuppet.getParentLayer().warperFrame(),
					matParentPuppet_Joint = mat3.multiply(matParentPuppet_Warper, mat3.multiply(warperRoot.remove, matWarper_Joint)),
					cLayerAttacher = subLayer.getDisplayContainer();

				cLayerAttacher.setMatrix(matParentPuppet_Joint);

			})(layer);

		}

		forEachWarperLayer(function (subLayer, tomSubLayerPrevPrev, tomSubLayerPrev, tomSubLayerNow) {
			warp(subLayer, tomSubLayerPrevPrev, tomSubLayerPrev, tomSubLayerNow);
		})(layer, tomPrevPrev, tomPrev, tomNow);
	},
};

return LayerMatrix;
})();
} else { // TO BE REMOVED --------------------------------------------------------------------------------------
 return (function () {
/*============================================================================
	Helpers
============================================================================*/

function valueFixedPoint (tomPrev0, lomVal) {
	if (tomPrev0) {
		return tomPrev0.get("value") === lomVal;
	}

	return false;
}

function childrenFixedPoint (tomPrev0, tomKids) {
	if (tomPrev0) {
		return tomPrev0.get("children") === tomKids;
	}

	return false;
}

// FIXME: if it shows up in the profiler move into c++
function matrixFixedPoint (prev, now) {
	// Tuned for Gene by setting values that don't lead to return to rest pop.
	// That test shoulde be performed in the absence of all other behaviors.
	var	translationEps = 0.1,
		linearEps = 0.05;

	var delta;

	// translation
	delta = prev[6] - now[6];
	if (delta < -translationEps || delta > translationEps) return false;

	delta = prev[7] - now[7];
	if (delta < -translationEps || delta > translationEps) return false;

	// linear
	delta = prev[0] - now[0];
	if (delta < -linearEps || delta > linearEps) return false;

	delta = prev[1] - now[1];
	if (delta < -linearEps || delta > linearEps) return false;

	delta = prev[3] - now[3];
	if (delta < -linearEps || delta > linearEps) return false;

	delta = prev[4] - now[4];
	if (delta < -linearEps || delta > linearEps) return false;

	return true;
}

function propertyNamed (key) {
	return function (collection) { return collection.get(key); };
}

function forEachWarperLayer (fn) {
	// function (layer, tom_1, tom_2, ...) -> function (layer, [tom_1, tom_2, ...])
	var visit = lodash.spread(fn, 1);

	function walk (layer, toms) {
		var puppet = layer.getPuppet(),
			layers = puppet.getLayers(),
			tomKids = lodash.chain(toms).map(propertyNamed("children")),
			index = -1,
			size = layers.length;

		while (++index < size) {
			var subLayer = layers[index];
			if (subLayer.getPuppet()) {
			if (subLayer.getWarpWithParent()) {
					walk(subLayer, toms);
				} else {
					visit(subLayer, tomKids.map(propertyNamed(subLayer.getStageId())).value());
				}
			}
		}
	}

	// function (layer, [tom_1, tom_2, ...]) -> function (layer, tom_1, tom_2, ...)
	return lodash.rest(walk);
}

function dofMatrixFromHandle (handle) {
	return new dofs.Matrix({ 
		type : handle.getAutoAttribute(),
		matrix : mat3().setIdentity()
	});
}

function dofResetTypeFromHandle (handle) {
	return function (dof) {
		return dof.set("type", handle.getAutoAttribute());
	};
}

function isHandleReleased (tree, ref, type) {
	var handle = tree.aHandle[ref];
	// correct check for both free (kNotSet) handles and (Hinge, Weld) joint handles.
	return handle.getAutoAttribute() === type;
}

// Set enabled to true to report when Layer reaches fixed point and stops warping.
var reportLayerWarps = { enabled : false },
	tLayerState = {};

devInternal.ifEnabled(reportLayerWarps, {

	count (layer) {
		tLayerState[layer.getStageId()] += 1;
	},

	show (layer, aPuppetMatLeaf) {
		var warpCount = tLayerState[layer.getStageId()];
		tLayerState[layer.getStageId()] = 0;

		if (warpCount > 0) {
			console.log("layer '" + layer.getName() + "' stops:\n" + aPuppetMatLeaf);
		}
	}
});


function noop () {}
function trace (name, dst, src) {
	lodash.assignWith(dst, src, function (valDst, valSrc, keySrc) {
		if (dst.enabled === false) return noop;
		if (valDst === false) return noop;

		return function () {
			var report = name + " " + keySrc + "():\n";

			for (var i = 0; i < arguments.length; i++) {
				report += arguments[i] + "\n";
			}

			console.logToUser(report);
			valSrc.apply(this, arguments);
		};
	});
}

// Debugging aid for tracing execution within warper.
// Reports stages and arguments when when enabled and noop otherwise.
var traceWarp = { enabled : false };
trace("traceWARP", traceWarp, {
	begin : noop,
	layer : noop,
	skipLayer : console.logToUser,
	released : noop,
	fixedPoint : console.logToUser,
	warp : console.logToUser,
	kids : noop,
	skipKids : noop,

});


/*
 * The origin handle defines the root frame for warping by its descendent leaf handles.
 * Returned object contains (1) value: the motion of that root,
 * and (2) remove/add: the change of coordinates needed to remove and add that motion from/to descendent handles.
 * 
 * TODO: Long-term gatherHandleFrames (and related functions) should support this functionality.
 * For now, we patch it by explicitlity removing/adding to retain the invariant of compute these frames relative to the Puppet frame.
 */
function warperRootFrame (frameRoot) {
	var value = frameRoot.matrix,
		remove, add;

	try {
		remove = mat3.invert(value);
		add = value;

		return {
			value, add, remove
		};
	} catch (e) {
		// we cannot change coordinates to remove singular root frame
		return {
			value,
			add : false, // signals singular root frame
			remove : mat3.identity() // allows non-distructive removal i.e. when multplying with .remove value
		};
	}
}

var LayerMatrix = {

	init : function init (layer) {
		var lomValue = internal.listInit(layer, dofMatrixFromHandle);

		var tomKids = immutable.Map().withMutations(function (tom) {

			forEachWarperLayer(function (subLayer) {
				tom.set(subLayer.getStageId(), init(subLayer));
			})(layer);

		});

		return new dofs.Tree({
			value : lomValue,
			children : tomKids
		});
	},

	resetType : function resetType (layer, tomLayer) {
		var lomValue = internal.listUpdate(layer, tomLayer.get("value"), dofResetTypeFromHandle);
		var tomKids = immutable.Map().withMutations(function (tom) {

			forEachWarperLayer(function (subLayer, tomNow) {
				var tomNext = resetType(subLayer, tomNow);
				tom.set(subLayer.getStageId(), tomNext);
			})(layer, tomLayer);

		});

		return new dofs.Tree({
			value : lomValue,
			children : tomKids
		});
	},

	// Layer ToM? ToM ToM -> ToM
	warp : function warp (layer, tomPrev0, tomNow, tomNext) {

		traceWarp.begin(layer.getName());

		var lomNow = tomNow.get("value"),
			lomNext = tomNext.get("value");

		// initialize assuming warp won't change the new set of dofs
		var lomWarp = lomNext,
			aFrameNow = false,
			aFrameNext = false,
			warperRoot = false,
			warperRootValueNow = false;

		traceWarp.layer(tomPrev0 && tomPrev0.get("value"), lomNow, lomNext);

		// skip warp if at fixed point and there is no new work
		if ( valueFixedPoint(tomPrev0, lomNow) && (lomNow === lomNext) ) {

			traceWarp.skipLayer(tomPrev0 && tomPrev0.get("value"), lomNow, lomNext);

		} else {

			var puppet = layer.getPuppet(),
				cWarper = layer.getWarperContainer();

			utils.assert(puppet && cWarper, "no warping container.");

			var	tree = layer.getHandleTreeArray(),
				aLeafRef = tree.getLeafRefArray();

			aFrameNow = internal.gatherHandleFrames(tree, lomNow);
			aFrameNext = internal.gatherHandleFrames(tree, lomNext);

			// Same goals as code in the comments but faster performance:
			// var	aMatLeaf = lodash(aPuppetFrames).at(aLeafRef).map(lodash.property("matrix")).value(),
			// 		aTypeLeaf = lodash(aPuppetFrames).at(aLeafRef).map(lodash.property("type")).value();
			var	nLeaf = aLeafRef.length,
				nReleased = 0,
				aMatLeafNext = new Array(nLeaf),
				aMatAnchorNext = new Array(nLeaf),
				aTypeLeafNext = new Array(nLeaf),

				aMatLeafNow = new Array(nLeaf);

			aLeafRef.forEach(function (ref, i) {
				var frameNext = aFrameNext[ref];

				aMatLeafNext[i] = frameNext.matrix;
				aMatAnchorNext[i] = frameNext.anchor;
				aTypeLeafNext[i] = frameNext.type;

				aMatLeafNow[i] = aFrameNow[ref].matrix;

				if (isHandleReleased(tree, ref, aTypeLeafNext[i])) nReleased += 1;
			});

			warperRoot = warperRootFrame(aFrameNext[0]);

			// Record current matrix for later checking whether root (i.e. non-leaf) dof changes.
			warperRootValueNow = cWarper.getMatrix();
			cWarper.setMatrix(warperRoot.value);

			// skip warp for singular root frame
			if (warperRoot.add !== false) {
				// temporarily remove warper root value from leaf handles...
				var	aMat_Handle = aMatLeafNext.map(function (matContainer_Handle) {
						return mat3.multiply(warperRoot.remove, matContainer_Handle);
					});

				cWarper.warp(aMat_Handle, aTypeLeafNext);

				// ...and add back warper root value so that state mirrors accumulated values of all handles
				aMat_Handle.forEach(function (mi, i) {
					mat3.multiply(warperRoot.add, mi, aMatLeafNext[i]);
				});

				// check if warp reached a fixed point -- leaf handles don't change much
				// FIXME: if this shows up in the profiler move to c++
				var index = nLeaf,
					atFixedPoint = true;
				while (--index >= 0)	{
					if ( !matrixFixedPoint(aMatLeafNow[index], aMatLeafNext[index]) ) {
						atFixedPoint = false;
						break;
					}
				}

				// scatter to update leaf handles iff not at fixed point.
				if ( atFixedPoint ) {
					// assume no change
					traceWarp.fixedPoint(aMatLeafNow, aMatLeafNext);
				} else {
					// fast special purpose scatter that reuses already computed values for leaf parents
					lomWarp = lomNext.withMutations(lodash.partial(internal.scatterLeafFrames, tree, aFrameNext, aMatLeafNext));
					traceWarp.warp(aMatLeafNow, aMatLeafNext, lomWarp);
				}
			}
		}

		// Update attachment cooordinates when when handle dofs change.
		// The first condition takes care of initialization case: container matrix does not yet mirror the value of the root handle.
		if ( (warperRootValueNow && !mat3.equalsApproximately(warperRootValueNow, warperRoot.value)) || (lomWarp !== lomNext) ) {

			forEachWarperLayer(function (subLayer) {

				var matWarper_Joint;

				const joint = getJointHandle(subLayer);

				if (joint) {
					// Calculate sub-puppet position from attach-to handle.
					const tree = layer.getHandleTreeArray(),
						jointRef = tree.getHandleRef(joint);

					matWarper_Joint = aFrameNext[jointRef].matrix;
				} else {
					// Calculate sub-puppet position from attach-to coordinates.
					matWarper_Joint = layer.getWarpAtAttachment(aMatLeafNext, subLayer);
				}

				subLayer.filterJointFrame(matWarper_Joint, matWarper_Joint);

				var parentPuppet = subLayer.parentPuppet,
					matParentPuppet_Warper = parentPuppet.getParentLayer().warperFrame(),
					matParentPuppet_Joint = mat3.multiply(matParentPuppet_Warper, mat3.multiply(warperRoot.remove, matWarper_Joint)),
					cLayerAttacher = subLayer.getDisplayContainer();

				cLayerAttacher.setMatrix(matParentPuppet_Joint);

			})(layer);

		}

		var tomKidsNow = tomNow.get("children"),
			tomKidsNext = tomNext.get("children"),
			tomKidsWarp = tomKidsNext;

		traceWarp.kids(tomPrev0 && tomPrev0.get("children"), tomKidsNow, tomKidsNext);

		// skip kid warps without new work and kids are at fixed point
		if ( childrenFixedPoint(tomPrev0, tomKidsNow) && (tomKidsNow === tomKidsNext) ) {

			traceWarp.skipKids(tomPrev0 && tomPrev0.get("children"), tomKidsNow, tomKidsNow);

		} else {

			tomKidsWarp = tomKidsNext.withMutations(function (tom) {

				if (tomPrev0) {
					var tomPrev = tomPrev0;
					forEachWarperLayer(function (subLayer, tomSubLayerPrev, tomSubLayerNow, tomSubLayerNext) {

						var tomSubLayerWarp = warp(subLayer, tomSubLayerPrev, tomSubLayerNow, tomSubLayerNext);
						if (tomSubLayerWarp !== tomSubLayerNext) tom.set(subLayer.getStageId(), tomSubLayerWarp);

					})(layer, tomPrev, tomNow, tomNext);
				} else {
					forEachWarperLayer(function (subLayer, tomSubLayerNow, tomSubLayerNext) {

						var tomSubLayerWarp = warp(subLayer, tomPrev0, tomSubLayerNow, tomSubLayerNext);
						if (tomSubLayerWarp !== tomSubLayerNext) tom.set(subLayer.getStageId(), tomSubLayerWarp);

					})(layer, tomNow, tomNext);
				}

			});

		}

		return tomNext.withMutations(function (tom) {
			if (lomWarp !== lomNext) tom.set("value", lomWarp);
			if (tomKidsWarp !== tomKidsNext) tom.set("children", tomKidsWarp);
		});
	},
};

return LayerMatrix;
})();


}
}); // end define


